document.addEventListener('DOMContentLoaded', () => {
    const productDetailContainer = document.getElementById('productDetailContainer');
    const cartItemCountSpan = document.getElementById('cartItemCount');
    const toastContainer = document.getElementById('toastContainer');
    const randomProductsGrid = document.getElementById('randomProductsGrid'); // Get the random products grid element

    let selectedProduct = null;
    let selectedColor = null;
    let selectedSize = null;
    let currentQuantity = 1;

    function showToast(message, duration = 3000) {
        const toast = document.createElement('div');
        toast.classList.add('toast-message');
        toast.textContent = message;
        toastContainer.appendChild(toast);
        setTimeout(() => {
            toast.remove();
        }, duration + 500);
    }

    function getCartFromLocalStorage() {
        const cartString = localStorage.getItem('cart');
        return cartString ? JSON.parse(cartString) : [];
    }

    function saveCartToLocalStorage(cart) {
        localStorage.setItem('cart', JSON.stringify(cart));
    }

    function updateCartItemCount() {
        const cart = getCartFromLocalStorage();
        const totalItems = cart.reduce((sum, item) => sum + item.quantity, 0);
        cartItemCountSpan.textContent = totalItems;
    }

    function addToCart(productId, color, size, quantity = 1) { // Default quantity to 1 for random products
        const products = JSON.parse(localStorage.getItem('allProducts') || '[]');
        const product = products.find(p => p.id === productId);

        if (product) {
            let cart = getCartFromLocalStorage();
            const uniqueItemId = `${productId}-${color || 'no-color'}-${size || 'no-size'}`;
            const existingItemIndex = cart.findIndex(item => item.uniqueId === uniqueItemId);

            if (existingItemIndex > -1) {
                cart[existingItemIndex].quantity += quantity;
            } else {
                cart.push({
                    uniqueId: uniqueItemId,
                    id: product.id,
                    name: product.name,
                    price: product.price,
                    image: product.variants.find(v => v.color === color)?.image || product.image,
                    color: color,
                    size: size,
                    quantity: quantity
                });
            }
            saveCartToLocalStorage(cart);
            updateCartItemCount();
            showToast(`Đã thêm "${product.name} (${color || ''} - ${size || ''})" vào giỏ hàng!`);
        }
    }

    function updateSizeOptions() {
        const sizeOptionsDiv = document.getElementById('sizeOptions');
        if (!sizeOptionsDiv) {
            console.warn('Element with ID "sizeOptions" not found. Cannot update sizes.');
            return;
        }

        sizeOptionsDiv.innerHTML = '';

        if (!selectedProduct || !selectedProduct.variants || selectedProduct.variants.length === 0) {
            selectedSize = 'Freesize';
            sizeOptionsDiv.innerHTML = `<button class="selected" data-size="Freesize">Freesize</button>`;
            return;
        }

        const currentVariant = selectedProduct.variants.find(v => v.color === selectedColor);

        if (currentVariant && currentVariant.sizes && currentVariant.sizes.length > 0) {
            if (!currentVariant.sizes.includes(selectedSize) || !selectedSize) {
                selectedSize = currentVariant.sizes[0];
            }

            currentVariant.sizes.forEach(size => {
                const isSelected = size === selectedSize ? 'selected' : '';
                const sizeButton = document.createElement('button');
                sizeButton.classList.add(isSelected);
                sizeButton.dataset.size = size;
                sizeButton.textContent = size;
                sizeButton.addEventListener('click', () => {
                    document.querySelectorAll('#sizeOptions button').forEach(btn => btn.classList.remove('selected'));
                    sizeButton.classList.add('selected');
                    selectedSize = sizeButton.dataset.size;
                });
                sizeOptionsDiv.appendChild(sizeButton);
            });
        } else {
            selectedSize = 'Freesize';
            sizeOptionsDiv.innerHTML = `<button class="selected" data-size="Freesize">Freesize</button>`;
        }
    }

    function displayProductDetail(product) {
        if (!product) {
            productDetailContainer.innerHTML = '<p>Không tìm thấy sản phẩm.</p>';
            return;
        }

        selectedProduct = product;

        if (product.variants && product.variants.length > 0) {
            selectedColor = product.variants[0].color;
            selectedSize = product.variants[0].sizes[0];
        } else {
            selectedColor = 'No Color';
            selectedSize = 'Freesize';
        }

        const mainImageSrc = product.variants.find(v => v.color === selectedColor)?.image || product.image;

        let colorOptionsHtml = '';
        if (product.variants && product.variants.length > 0) {
            product.variants.forEach(variant => {
                const isSelected = variant.color === selectedColor ? 'selected' : '';
                colorOptionsHtml += `<button class="${isSelected}" data-color="${variant.color}">${variant.color}</button>`;
            });
        } else {
            colorOptionsHtml = `<button class="selected" data-color="No Color">Không màu</button>`;
        }

        productDetailContainer.innerHTML = `
            <div class="product-detail-images">
                <img src="${mainImageSrc}" alt="${product.name}" class="main-image" id="mainProductImage">
                <div class="thumbnail-gallery" id="thumbnailGallery">
                </div>
            </div>
            <div class="product-detail-info">
                <h1>${product.name}</h1>
                <div class="price-section">
                   <span class="current-price">${product.price.toLocaleString('th-TH', { style: 'currency', currency: 'THB' })}
                   </span>
                </div>
                <div class="product-options">
                    ${product.variants && product.variants.length > 0 ? `
                        <div class="option-group color-options">
                            <label>สี:</label>
                            <div class="option-buttons" id="colorOptions">
                                ${colorOptionsHtml}
                            </div>
                        </div>
                        <div class="option-group size-options">
                            <label>ขนาด:</label>
                            <div class="option-buttons" id="sizeOptions">
                                </div>
                        </div>
                    ` : ''}
                </div>
                <div class="quantity-selector">
                    <label>จำนวน:</label>
                    <button id="decreaseQuantity">-</button>
                    <input type="number" id="productQuantity" value="${currentQuantity}" min="1" max="99" readonly>
                    <button id="increaseQuantity">+</button>
                </div>
                <button class="add-to-cart-btn" id="addToCartBtn">
                    <i class="fas fa-cart-plus"></i> ซื้อสินค้า
                </button>

            </div>
        `;

        const thumbnailGallery = document.getElementById('thumbnailGallery');
        if (product.variants && product.variants.length > 0) {
            product.variants.forEach(variant => {
                const img = document.createElement('img');
                img.src = variant.image;
                img.alt = variant.color;
                img.dataset.color = variant.color;
                if (variant.color === selectedColor) {
                    img.classList.add('active');
                }
                img.addEventListener('click', () => {
                    document.querySelectorAll('.thumbnail-gallery img').forEach(thumb => thumb.classList.remove('active'));
                    img.classList.add('active');
                    document.getElementById('mainProductImage').src = img.src;
                    selectedColor = img.dataset.color;
                    updateSizeOptions();
                });
                thumbnailGallery.appendChild(img);
            });
        } else {
            const img = document.createElement('img');
            img.src = product.image;
            img.alt = product.name;
            img.classList.add('active');
            img.addEventListener('click', () => {
                document.getElementById('mainProductImage').src = img.src;
            });
            thumbnailGallery.appendChild(img);
        }

        document.querySelectorAll('#colorOptions button').forEach(button => {
            button.addEventListener('click', () => {
                document.querySelectorAll('#colorOptions button').forEach(btn => btn.classList.remove('selected'));
                button.classList.add('selected');
                selectedColor = button.dataset.color;
                const newMainImage = selectedProduct.variants.find(v => v.color === selectedColor)?.image || selectedProduct.image;
                document.getElementById('mainProductImage').src = newMainImage;
                document.querySelectorAll('.thumbnail-gallery img').forEach(thumb => {
                    if (thumb.dataset.color === selectedColor) {
                        thumb.classList.add('active');
                    } else {
                        thumb.classList.remove('active');
                    }
                });
                updateSizeOptions();
            });
        });

        const quantityInput = document.getElementById('productQuantity');
        document.getElementById('decreaseQuantity').addEventListener('click', () => {
            currentQuantity = Math.max(1, currentQuantity - 1);
            quantityInput.value = currentQuantity;
        });
        document.getElementById('increaseQuantity').addEventListener('click', () => {
            currentQuantity = Math.min(99, currentQuantity + 1);
            quantityInput.value = currentQuantity;
        });
        quantityInput.addEventListener('change', () => {
            let val = parseInt(quantityInput.value);
            if (isNaN(val) || val < 1) val = 1;
            if (val > 99) val = 99;
            currentQuantity = val;
            quantityInput.value = currentQuantity;
        });

        document.getElementById('addToCartBtn').addEventListener('click', () => {
            if (!selectedColor || !selectedSize) {
                showToast('Vui lòng chọn màu sắc và kích cỡ!');
                return;
            }
            addToCart(product.id, selectedColor, selectedSize, currentQuantity);
        });
    }

    const urlParams = new URLSearchParams(window.location.search);
    const productId = parseInt(urlParams.get('id'));

    if (productId) {
        const products = JSON.parse(localStorage.getItem('allProducts') || '[]');
        const product = products.find(p => p.id === productId);
        displayProductDetail(product);
        updateSizeOptions();
    } else {
        productDetailContainer.innerHTML = '<p>Không có sản phẩm nào được chọn.</p>';
    }

    // --- NEW FUNCTION: Display Random Products ---
    function displayRandomProducts() {
        const allProducts = JSON.parse(localStorage.getItem('allProducts') || '[]');
        if (allProducts.length === 0) {
            if (randomProductsGrid) { // Check if randomProductsGrid exists
                randomProductsGrid.innerHTML = '<p>ไม่มีสินค้าที่แนะนำ</p>'; // No recommended products
            }
            return;
        }

        // Shuffle products and pick 4 random ones
        const shuffledProducts = allProducts.sort(() => 0.5 - Math.random());
        const productsToDisplay = shuffledProducts.slice(0, 4); // Get the first 4

        if (randomProductsGrid) { // Check if randomProductsGrid exists
            randomProductsGrid.innerHTML = ''; // Clear previous content

            productsToDisplay.forEach(product => {
                const productItem = document.createElement('div');
                productItem.classList.add('product-item'); // Reuse product-item class for styling
                productItem.innerHTML = `
                    <div class="product-image-wrapper">
                        <img src="${product.image}" alt="${product.name}">
                    </div>
                    <div class="product-info">
                        <h3>${product.name}</h3>
                        <p class="price">${product.price.toLocaleString('th-TH', { style: 'currency', currency: 'THB' })}</p>
                        <button class="add-to-cart-btn-random" data-id="${product.id}">เพิ่มลงรถเข็น</button> </div>
                `;
                randomProductsGrid.appendChild(productItem);
            });

            // Add event listeners for "Add to Cart" buttons on random products
            document.querySelectorAll('.add-to-cart-btn-random').forEach(button => {
                button.addEventListener('click', (e) => {
                    e.stopPropagation();
                    const productId = parseInt(e.target.dataset.id);
                    const product = allProducts.find(p => p.id === productId);

                    if (product) {
                        let defaultColor = 'N/A';
                        let defaultSize = 'N/A';
                        if (product.variants && product.variants.length > 0) {
                            defaultColor = product.variants[0].color || 'N/A';
                            defaultSize = product.variants[0].sizes && product.variants[0].sizes.length > 0 ? product.variants[0].sizes[0] : 'N/A';
                        } else {
                            // If no variants, ensure color and size are set for uniqueId
                            defaultColor = 'No Color';
                            defaultSize = 'Freesize';
                        }
                        addToCart(product.id, defaultColor, defaultSize);
                    }
                });
            });

            // Add click listener for product items to navigate to detail page
            document.querySelectorAll('.random-products-section .product-item').forEach(item => {
                item.addEventListener('click', (e) => {
                    if (e.target.classList.contains('add-to-cart-btn-random')) {
                        return;
                    }
                    const productId = parseInt(item.querySelector('.add-to-cart-btn-random').dataset.id);
                    window.location.href = `product-detail.html?id=${productId}`;
                });
            });
        }
    }
    updateCartItemCount();
    displayRandomProducts(); // Call the function to display random products
});